package ga.core.individual.population;

import ga.core.evaluation.IFitnessEvaluator;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.validation.GAContext;
import ga.core.validation.IValidator;

import java.util.Iterator;

/**
 * Common interface for populations of a GA.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public interface IPopulation<T extends IIndividual<T>> extends Iterable<T> {

  /**
   * Fill the population with random individuals.
   * 
   * @param validator
   *          The validator or <code>null</code>.
   * @param context
   *          The GA context.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void initRandomly(IValidator<T> validator, GAContext context);

  /**
   * Sets the evaluator.
   * 
   * @param evaluator
   *          The evaluator.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void setEvaluator(IFitnessEvaluator<T> evaluator);

  /**
   * Getter for a list representation of the population.
   * 
   * @return The individuals as list.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IndividualList<T> getIndividuals();

  /**
   * Adds individuals to the population.
   * 
   * @param individuals
   *          Individuals to add.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void addIndividuals(T... individuals);

  /**
   * Adds an individual to the population.
   * 
   * @param individual
   *          Individual to add.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void addIndividual(T individual);

  /**
   * Adds individuals to the population.
   * 
   * @param individuals
   *          Individuals to add.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void addIndividuals(IndividualList<T> individuals);

  /**
   * Clears the population.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void clear();

  /**
   * Evaluates all individiduals using the automatic evaluator.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public void evaluateAutomatic();

  /**
   * Determines if the population is empty.
   * 
   * @return <code>true</code> if the population is empty.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public boolean isEmpty();

  /**
   * Returns the size of the population; the count of individuals.
   * 
   * @return Size.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int size();

  /**
   * Gets the individual if the lowest fitness.
   * 
   * @return The unfittest individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public T getUnfittestIndividual();

  /**
   * Gets the individual if the highest fitness.
   * 
   * @return The fittest individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public T getFittestIndividual();

  /**
   * Gets the elite individual, if elite strategy is used.
   * 
   * @return The elite or <code>null</code>.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public T getEliteIndividual();

  /**
   * Randomly selects an individual to evaluate.
   * 
   * @return Random individual to evaluate.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public T getRandomIndividualForEvaluation();

  /**
   * Get all individuals that has not been evaluated yet.
   * 
   * @return List of unevaluated individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public IndividualList<T> getUnevaluatedIndividuals();

  /**
   * Get the number of individuals that has been evaluated.
   * 
   * @return Number of evaluated individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public int getEvaluatedIndividualCount();

  /**
   * Get the number of initial individuals.
   * 
   * @return Number of init individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public abstract int getInitIndividualCount();

  /**
   * Set the number of initial individuals.
   * 
   * @param individualCount
   *          Number of init individuals.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public abstract void setInitIndividualCount(final int individualCount);

  /**
   * Determines if duplicates are allowed.
   * 
   * @return <code>true</code> if duplicates are allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public boolean isAllowDuplicates();

  /**
   * Determine if one of the individuals is contained in this list.
   * 
   * @param list
   *          Individuals to look for.
   * @return <code>true</code> if this list contains any individual of the other
   *         one.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public boolean containsAny(IndividualList<T> list);

  @Override
  public Iterator<T> iterator();

  /**
   * Randomly selects an individual to select.
   * 
   * @return Random individual to select.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public T getRandomIndividualForSelection();
}
